const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const { validationResult } = require('express-validator');
const User = require('../models/User');
const logger = require('../utils/logger');
const { sendAdminCreationEmail, sendResellerCreationEmail } = require('../services/mailService');

exports.createAdmin = async (req, res) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) return res.status(400).json({ errors: errors.array() });

  try {
    const { name, email, password } = req.body;
    const adminKey = req.headers['x-admin-key'];

    const user = await User.findOne({ where: { email } });
    if (user) return res.status(400).json({ error: 'User already exists.' });
    if (adminKey !== process.env.ADMIN_KEY) {
      return res.status(400).json({ error: 'Invalid Admin Key!' });
    }

    await User.create({ name, email, password, accountType: 'admin', emailVerified: true });
    sendAdminCreationEmail(email);
    logger.info(`New admin registered: ${email}`);
    res.status(201).json({ message: 'Admin registered successfully' });
  } catch (error) {
    logger.error(`Registration error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};

exports.createReseller = async (req, res) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) return res.status(400).json({ errors: errors.array() });

  try {
    const { name, email, password } = req.body;

    const user = await User.findOne({ where: { email } });
    if (user) return res.status(400).json({ error: 'User already exists.' });

    const verificationToken = jwt.sign({ email }, process.env.JWT_SECRET, { expiresIn: '1d' });
    const verificationLink = `${process.env.FRONTEND_URL}/verify-email/${verificationToken}`;
    TokenExpiresIn = new Date(Date.now() + 60 * 60 * 1000).toISOString();
    await User.create({ name, email, password, accountType: 'reseller', verificationToken, TokenExpiresIn });
    sendResellerCreationEmail(email, verificationLink);
    res.status(201).json({ message: 'Reseller User registered successfully' });
  } catch (error) {
    logger.error(`Registration error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};


exports.login = async (req, res) => {
  try {
    const { email, password } = req.body;
    const user = await User.findOne({ where: { email } });
    if (!user) return res.status(400).json({ error: 'User not Found' });
    if (user.status === 'suspended' || user.status === 'blocked') {
      return res.status(400).json({ error: `Your account has been ${user.status}. Contact support for more information.` });
    }
    if (user.password === null && user.googleTokens) {
      return res.status(400).json({ error: 'Login with google to continue.' });
    }

    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) return res.status(400).json({ error: 'Invalid password' });
    const accountType = user.accountType;

    const token = jwt.sign({ id: user.id }, process.env.JWT_SECRET, { expiresIn: process.env.TOKEN_EXPIRE_TIME });
    res.json({ token, accountType });
  } catch (error) {
    logger.error(`Login error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};

exports.generateApiKey = async (req, res) => {
  try {
    const { id } = req.user;
    const user = await User.findByPk(id);

    if (!user || user.accountType !== 'application') {
      return res.status(403).json({ error: 'Only application users can generate API keys' });
    }

    const cryptoKey = crypto.randomBytes(32).toString('hex');
    const apiKey = `SK_${cryptoKey}`;
    const hashedKey = await bcrypt.hash(apiKey, 10); 

    user.apiKey = hashedKey;
    await user.save();

    res.json({ apiKey }); 
  } catch (error) {
    logger.error(`API Key generation error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};


// Update User
exports.updateUser = async (req, res) => {
  const { id } = req.user;
  const { name, email, password } = req.body;

  try {
    const user = await User.findByPk(id);
    if (!user) return res.status(404).json({ error: 'User not found' });

    if (name) user.name = name;
    if (email) user.email = email;
    if (password) user.password = await bcrypt.hash(password, 10);

    await user.save();
    res.json({ message: 'User updated successfully', user });
  } catch (error) {
    logger.error(`Update error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};

// Delete User
exports.deleteUser = async (req, res) => {
  const { id } = req.user;

  try {
    const user = await User.findByPk(id);
    if (!user) return res.status(404).json({ error: 'User not found' });

    await user.destroy();
    res.json({ message: 'User deleted successfully' });
  } catch (error) {
    logger.error(`Deletion error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};

// Fetch User Profile
exports.getProfile = async (req, res) => {
  const { id } = req.user;

  try {
    const user = await User.findByPk(id, {
      attributes: ['id', 'name', 'email', 'accountType', 'emailVerified', 'googleTokens', 'createdAt', 'updatedAt'],
    });

    if (!user) return res.status(404).json({ error: 'User not found' });

    res.json({ user });
  } catch (error) {
    logger.error(`Profile fetch error: ${error.message}`);
    res.status(500).json({ error: error.message });
  }
};