const axios = require('axios');
const fs = require('fs');
const path = require('path');
const logger = require('../utils/logger');
const Video = require('../models/Video');
const staticVoices = require('../data/voices.json');

exports.getVoices = async (req, res) => {
  try {
    // First try to get voices from ElevenLabs API
    const response = await axios.get('https://api.elevenlabs.io/v1/voices', {
      timeout: 10000 // Add timeout to prevent hanging
    });
    
    // If successful, return the API data
    return res.status(200).json(response.data);
  } catch (error) {
    logger.error(`Error fetching voices from ElevenLabs API: ${error.message}`);
    
    try {
      // If API call fails, return static voices data
      logger.info('Falling back to static voices data');
      return res.status(200).json(staticVoices);
    } catch (fallbackError) {
      logger.error(`Error with static voices fallback: ${fallbackError.message}`);
      return res.status(500).json({ 
        message: 'Failed to fetch voices and fallback failed',
        error: error.message
      });
    }
  }
};

exports.getArtStyles = async (req, res) => {
    try {
        const assetsDir = path.join(__dirname, '..', 'public', 'assets');

        const artStyles = [
            { name: '3D Render', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212071/autoshorts_vp1bzw.png', uuid: 'debdf72a-91a4-467b-bf61-cc02bdeb69c6' },
            { name: 'Graphic Design 3D', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212241/lego_cdyp9d.png', uuid: '7d7c2bc5-4b12-4ac3-81a9-630057e9e89f' },
            { name: 'Creative', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212168/comic_book_etbc5w.png', uuid: '6fedbf1f-4a17-45ec-84fb-92fe524a29ef' },
            { name: 'Dynamic', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212210/disney_toon_mjciwj.png', uuid: '111dc692-d470-4eec-b791-3475abac4c46' },
            { name: 'Fashion', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212372/studio_ghibli_zhyu4d.png', uuid: '594c4a08-a522-4e0e-b7ff-e4dac4b6b622' },
            { name: 'Game Concept', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212156/childrens_book_b3uzrb.png', uuid: '09d2b5b5-d7c5-4c02-905d-9f84051640f4' },
            { name: 'Illustration', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212359/photo_realism_glbprm.png', uuid: '645e4195-f63d-4715-a3f2-3fb1e6eb8c70' },
            { name: 'Portrait', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212351/minecraft_zpkvwu.png', uuid: '8e2bc543-6ee2-45f9-bcd9-594b6ce84dcd' },
            { name: 'WATERCOLOR', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212390/watercolor_hcuijh.png', uuid: '1db308ce-c7ad-4d10-96fd-592fa6b75cc4' },
            { name: 'Portrait Cinematic', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212228/horror_stmfrb.png', uuid: '4edb03c9-8a26-4041-9d01-f85b5d4abd71' },
            { name: 'Ray Traced', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212092/charcoal_zeau8y.png', uuid: 'b504f83c-3326-4947-82e1-7fe9e839ec0f' },
            { name: 'Stock Photo', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212218/gtav_bs0ocq.png', uuid: '5bdc3f2a-1be6-4d1c-8e77-992a30824a2c' },
            { name: 'ANIME', image: 'https://res.cloudinary.com/dxqooqe9h/image/upload/v1742212011/anime_lbjo2a.png', uuid: 'b2a54a51-230b-4d4f-ad4e-8409bf58645f' }
        ];

        res.status(200).json(artStyles);
    } catch (error) {
        logger.error(`Error processing art styles: ${error.message}`);
        res.status(500).json({ message: 'Failed to process art styles', error: error.message });
    }
};

exports.getUserVideos = async (req, res) => {
    try {
        const userId = req.user.id;

        const videos = await Video.findAll({
            where: { userId },
            order: [['createdAt', 'DESC']],
        });

        res.status(200).json(videos);
    } catch (error) {
        res.status(500).json({
            error: 'Failed to fetch videos',
            details: error.message,
        });
    }
}

exports.deleteVideo = async (req, res) => {
    try {
        const { id } = req.params;
        const userId = req.user.id; 

        const video = await Video.findOne({ where: { id } });

        if (!video) {
            return res.status(404).json({ message: 'Video not found' });
        }

        if (video.userId !== userId) {
            return res.status(403).json({ message: 'You are not authorized to delete this video' });
        }

        await video.destroy();

        res.status(200).json({ message: 'Video deleted successfully' });
    } catch (error) {
        logger.error(`Error deleting video: ${error.message}`);
        res.status(500).json({
            message: 'Failed to delete video',
            details: error.message,
        });
    }
};

