const { generateSecurePassword } = require('../utils/passwordGenerator');
const { sendAccountCreationEmail } = require('../services/mailService');
const User = require('../models/User');
const logger = require('../utils/logger');

const getPaginationParams = (query) => {
    const page = parseInt(query.page, 10) || 1;
    const limit = parseInt(query.limit, 10) || 10;
    const offset = (page - 1) * limit;
    return { limit, offset, page };
};

class UserManagementController {
    static async createUser(req, res) {
        try {
            const { id } = req.user;
            const creator = await User.findByPk(id);
            const { name, email, accountType } = req.body;
            const allowedAccountTypes = ['bundle', 'premium', 'reseller', 'application'];
            if (!['admin', 'reseller'].includes(creator.accountType)) {
                return res.status(403).json({ error: 'Unauthorized to create users' });
            }
            if (creator.accountType === 'reseller' && !['bundle', 'premium'].includes(accountType)) {
                return res.status(403).json({ error: 'Resellers can only create bundle or premium accounts' });
            }

            if (creator.accountType === 'reseller' && accountType === 'reseller') {
                return res.status(403).json({ error: 'Resellers cannot create other reseller accounts' });
            }
            if (!allowedAccountTypes.includes(accountType)) {
                return res.status(400).json({ error: `Invalid Account Type. Allowed types: ${allowedAccountTypes.join(', ')}` });
            }

            const user = await User.findOne({ where: { email } });
            if (user) return res.status(400).json({ error: 'User already exists.' });


            const password = generateSecurePassword();

            await User.create({
                name,
                email,
                password,
                accountType,
                createdBy: creator.id
            });

            await sendAccountCreationEmail(creator.email, email, password);

            logger.info(`New user created by ${creator.email}: ${email}`);
            res.status(201).json({
                message: 'User created successfully'
            });

        } catch (error) {
            logger.error(`User creation error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async updateUser(req, res) {
        try {
            const { id } = req.user;
            const creator = await User.findByPk(id);
            const { userId } = req.params;
            const updates = req.body;

            const userToUpdate = await User.findByPk(userId);

            if (!userToUpdate) {
                return res.status(404).json({ error: 'User not found' });
            }

            if (creator.accountType === 'reseller') {
                if (userToUpdate.createdBy !== creator.id) {
                    return res.status(403).json({ error: 'Unauthorized to update this user' });
                }

                if (updates.accountType && !['bundle', 'premium'].includes(updates.accountType)) {
                    return res.status(403).json({ error: 'Cannot upgrade to this account type' });
                }
            }

            if (creator.accountType === 'admin' && userToUpdate.accountType === 'admin') {
                return res.status(403).json({ error: 'Cannot update admin accounts' });
            }

            await userToUpdate.update(updates);


            logger.info(`User ${userId} updated by ${creator.email}`);
            res.json({
                message: 'User updated successfully',
                user: {
                    id: userToUpdate.id,
                    name: userToUpdate.name,
                    email: userToUpdate.email,
                    accountType: userToUpdate.accountType
                }
            });

        } catch (error) {
            logger.error(`User update error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getUser(req, res) {
        try {
            const { id } = req.user;
            const creator = await User.findByPk(id);
            const { userId } = req.params;

            const user = await User.findByPk(userId, {
                attributes: ['id', 'name', 'email', 'status', 'accountType', 'createdBy']
            });

            if (!user) {
                return res.status(404).json({ error: 'User not found' });
            }

            if (creator.accountType === 'reseller') {
                if (user.createdBy !== creator.id) {
                    return res.status(403).json({ error: 'Unauthorized to fetch this user' });
                }

            }
            res.json({
                message: 'User fetched successfully',
                user
            });

        } catch (error) {
            logger.error(`User update error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getResellerUsers(req, res) {
        try {
            const { limit, offset, page } = getPaginationParams(req.query);

            const { count, rows: users } = await User.findAndCountAll({
                where: {
                    createdBy: req.user.id
                },
                attributes: ['id', 'name', 'email', 'accountType', 'status', 'createdAt'],
                order: [['createdAt', 'DESC']],
                limit,
                offset
            });

            const totalPages = Math.ceil(count / limit);

            res.json({
                users,
                pagination: {
                    totalItems: count,
                    totalPages,
                    currentPage: page,
                    itemsPerPage: limit,
                    hasNextPage: page < totalPages,
                    hasPreviousPage: page > 1
                }
            });
        } catch (error) {
            logger.error(`Fetch reseller users error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getBundleUsers(req, res) {
        try {
            const { limit, offset, page } = getPaginationParams(req.query);
            const { id } = req.user;
            const user = await User.findByPk(id);

            const whereClause = {
                accountType: 'bundle'
            };

            if (user.accountType === 'reseller') {
                whereClause.createdBy = user.id;
            }

            const { count, rows: users } = await User.findAndCountAll({
                where: whereClause,
                attributes: ['id', 'name', 'email', 'accountType', 'status', 'createdAt'],
                order: [['createdAt', 'DESC']],
                limit,
                offset
            });

            const totalPages = Math.ceil(count / limit);

            res.json({
                users,
                pagination: {
                    totalItems: count,
                    totalPages,
                    currentPage: page,
                    itemsPerPage: limit,
                    hasNextPage: page < totalPages,
                    hasPreviousPage: page > 1
                }
            });
        } catch (error) {
            logger.error(`Fetch bundle users error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getPremiumUsers(req, res) {
        try {
            const { limit, offset, page } = getPaginationParams(req.query);
            const { id } = req.user;
            const user = await User.findByPk(id);

            const whereClause = {
                accountType: 'premium'
            };

            if (user.accountType === 'reseller') {
                whereClause.createdBy = user.id;
            }

            const { count, rows: users } = await User.findAndCountAll({
                where: whereClause,
                attributes: ['id', 'name', 'email', 'accountType', 'status', 'createdAt'],
                order: [['createdAt', 'DESC']],
                limit,
                offset
            });

            const totalPages = Math.ceil(count / limit);

            res.json({
                users,
                pagination: {
                    totalItems: count,
                    totalPages,
                    currentPage: page,
                    itemsPerPage: limit,
                    hasNextPage: page < totalPages,
                    hasPreviousPage: page > 1
                }
            });
        } catch (error) {
            logger.error(`Fetch premium users error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getResellerAccounts(req, res) {
        try {
            const { limit, offset, page } = getPaginationParams(req.query);
            const { id } = req.user;
            const user = await User.findByPk(id);

            if (user.accountType !== 'admin') {
                return res.status(403).json({ error: 'Unauthorized access' });
            }

            const { count, rows: users } = await User.findAndCountAll({
                where: {
                    accountType: 'reseller'
                },
                attributes: ['id', 'name', 'email', 'accountType', 'status', 'createdAt'],
                order: [['createdAt', 'DESC']],
                limit,
                offset
            });

            const totalPages = Math.ceil(count / limit);

            res.json({
                users,
                pagination: {
                    totalItems: count,
                    totalPages,
                    currentPage: page,
                    itemsPerPage: limit,
                    hasNextPage: page < totalPages,
                    hasPreviousPage: page > 1
                }
            });
        } catch (error) {
            logger.error(`Fetch reseller accounts error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }

    static async getAllUsers(req, res) {
        try {
            const { limit, offset, page } = getPaginationParams(req.query);
            const { id } = req.user;
            const user = await User.findByPk(id);

            if (user.accountType !== 'admin') {
                return res.status(403).json({ error: 'Unauthorized access' });
            }

            const { count, rows: users } = await User.findAndCountAll({
                where: {
                    accountType: ['bundle', 'premium', 'reseller']
                },
                attributes: ['id', 'name', 'email', 'accountType', 'status', 'createdAt'],
                order: [['createdAt', 'DESC']],
                limit,
                offset
            });

            const totalPages = Math.ceil(count / limit);

            res.json({
                users,
                pagination: {
                    totalItems: count,
                    totalPages,
                    currentPage: page,
                    itemsPerPage: limit,
                    hasNextPage: page < totalPages,
                    hasPreviousPage: page > 1
                }
            });
        } catch (error) {
            logger.error(`Fetch all users error: ${error.message}`);
            res.status(500).json({ error: error.message });
        }
    }
}

module.exports = UserManagementController;