const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const User = require('../models/User');
const { user } = require('elevenlabs/api');

const authMiddleware = (req, res, next) => {
  const authHeader = req.header('Authorization');
  if (!authHeader) {
    return res.status(401).json({ error: 'Access denied, no token provided' });
  }

  const token = authHeader.split(' ')[1];
  if (!token) {
    return res.status(401).json({ error: 'Access denied, invalid token format' });
  }

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (error) {
    res.status(401).json({ error: 'Invalid token' });
  }
};

const apiKeyMiddleware = async (req, res, next) => {
  const apiKey = req.header('x-api-key');

  if (!apiKey) {
    return res.status(401).json({ error: 'API key is required' });
  }

  try {
    const users = await User.findAll({ where: { accountType: 'application' } });
    

    const matchedUser = await Promise.any(users.map(async (user) => {
      const isMatch = await bcrypt.compare(apiKey, user.apiKey || '');
      return isMatch ? user : Promise.reject();
    }));

    req.user = { id: matchedUser.id, accountType: matchedUser.accountType };
    next();
  } catch (error) {
    return res.status(401).json({ error: 'Invalid API key' });
  }
};

const isAdminOrReseller = async (req, res, next) => {
  const { id } = req.user;
  const user = await User.findByPk(id);
  if (!['admin', 'reseller'].includes(user.accountType)) {
    return res.status(403).json({ error: 'Not Allowed' });
  }
  next();
};

const isAdmin = async (req, res, next) => {
  const { id } = req.user;
  const user = await User.findByPk(id);
  if (user.accountType !== 'admin') {
    return res.status(403).json({ error: 'Admin access required' });
  }
  next();
};

const isReseller = async (req, res, next) => {
  const { id } = req.user;
  const user = await User.findByPk(id);
  if (user.accountType !== 'reseller') {
    return res.status(403).json({ error: 'Reseller access required' });
  }
  next();
};

module.exports = {
  authMiddleware,
  isAdminOrReseller,
  isAdmin,
  isReseller,
  apiKeyMiddleware
};
