const fs = require('fs').promises;
const path = require('path');
const logger = require('../utils/logger');

class TempCleanupService {
    /**
     * Cleans up all files and subdirectories within a given directory.
     * @param {string} directoryPath - The path of the directory to clean up.
     */
    async cleanUp(directoryPath) {
        try {
            const entries = await fs.readdir(directoryPath, { withFileTypes: true });

            for (const entry of entries) {
                const fullPath = path.join(directoryPath, entry.name);

                if (entry.isDirectory()) {
                    // Recursively clean subdirectories
                    await this.cleanUp(fullPath);
                    await fs.rmdir(fullPath); // Remove the empty directory
                } else {
                    // Delete files
                    await fs.unlink(fullPath);
                }
            }

            logger.info(`Temporary files in ${directoryPath} have been cleaned up successfully.`);
        } catch (error) {
            if (error.code === 'ENOENT') {
                logger.warn(`Directory ${directoryPath} does not exist or has already been cleaned.`);
            } else {
                logger.error(`Error cleaning up temporary files in ${directoryPath}:`, error);
                throw error;
            }
        }
    }
}

module.exports = new TempCleanupService();
