
const OpenAIService = require('../services/openaiService');
const ElevenLabsService = require('../services/elevenLabsService');
const LeonardoService = require('../services/leonardoService');
const { parseElevenLabsTimestamps } = require('../utils/distributedTimestamps');
const { Op } = require('sequelize');
const User = require('../models/User');
const Video = require('../models/Video');
const logger = require('../utils/logger');
const processVideoGenerationJob = require('../services/videoService');
const generateVideoInBackground = require('../services/videoService');
const { alignment } = require('../alignment');

class VideoController {

    async generateInitial(req, res) {
        try {
            const { 
                voiceType, 
                artStyle, 
                aspectRatio, 
                duration, 
                platform, 
                customPrompt, 
                external = null,
                externalWebhook = null,
                generatedPrompt,
                uploadTime = null,
                enableCaptions = false,
                captionFont = null,
                captionFontSize = null,
                captionFontColor = null,
                captionPosition = null
            } = req.body;
           
            const { id } = req.user;
            const user = await User.findByPk(id);
            const email = user.email;

            if (user.accountType !== 'application' &&
                (external || externalWebhook)) {
                return res.status(403).json({ error: 'external requests not allowed' });
            }

            if (user.accountType === 'application' &&
                (!external || !externalWebhook)) {
                return res.status(403).json({ error: 'external requests required for application account type' });
            }

            if (user.accountType !== 'application') {
                  const existingJob = await Video.findOne({
                where: { userId: user.id, currentStep: { [Op.ne]: 'done' } } 
            });
            
            if (existingJob) {
                return res.status(400).json({ 
                    error: "You have an ongoing job. Please complete it before starting another."
                });
            }

            }
          
            // Validate required fields
            const requiredFields = [ 'voiceType', 'artStyle', 'aspectRatio', 'duration', 'platform'];
            const missingFields = requiredFields.filter((field) => !req.body[field]);

            if (missingFields.length > 0) {
                return res.status(400).json({
                    error: `Missing required fields: ${missingFields.join(', ')}`,
                });
            }

            const allowedCaptionColors = [
            'white', 'black', 'yellow', 'red', 'blue', 'green', 'cyan', 'magenta', 'orange'
            ];

            const allowedCaptionFonts = [
            'Arial', 'Helvetica', 'Verdana', 'Times New Roman', 'Courier New',
            'Georgia', 'Trebuchet MS', 'Impact', 'Tahoma'
            ];

            const allowedCaptionPositions = ['top', 'center', 'bottom'];

            const isValidPreset = (value, allowedList) => {
            return allowedList.includes(value);
            };

            let fontSize = 0;

            if (enableCaptions) {
                if (captionFontColor && !isValidPreset(captionFontColor.toLowerCase(), allowedCaptionColors)) {
                    return res.status(400).json({
                    error: `Invalid caption font color. Allowed values: ${allowedCaptionColors.join(', ')}`,
                    });
                }

                if (captionFont && !isValidPreset(captionFont, allowedCaptionFonts)) {
                    return res.status(400).json({
                    error: `Invalid caption font. Allowed fonts: ${allowedCaptionFonts.join(', ')}`,
                    });
                }

                if (captionPosition && !isValidPreset(captionPosition.toLowerCase(), allowedCaptionPositions)) {
                    return res.status(400).json({
                    error: `Invalid caption position. Allowed positions: ${allowedCaptionPositions.join(', ')}`,
                    });
                }

                fontSize = Number(captionFontSize);
                
                if (fontSize && (typeof fontSize !== 'number')) {
                    return res.status(400).json({
                    error: 'Caption font size must be a number between 12 and 100.',
                    });
                }
                }

            
            if (platform === "webhook" && !externalWebhook && !external) {
                return res.status(400).json({
                    error: `Missing required fields for webhook`,
                });
            }

            // Validate platform
            const validPlatforms = ['email', 'youtube', 'webhook', 'both'];
            if (!validPlatforms.includes(platform)) {
                return res.status(400).json({
                    error: `Invalid platform. Valid options are: ${validPlatforms.join(', ')}`,
                });
            }

            if (!email) {
                return res.status(401).json({ error: 'User email not found in token' });
            }

            // Determine the prompt to use (custom or generated)
            const promptToUse = customPrompt || generatedPrompt;

            if (!promptToUse) {
                return res.status(400).json({
                    error: 'Either customPrompt or generatedPrompt must be provided.',
                });
            }

            const jobId = `video-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;

            const captionStyleDetails = {
                font: captionFont,  
                fontSize: fontSize,
                fontColor: captionFontColor,
                position: captionPosition,
            };

            // Store the prompt in the database

            const video = await Video.create({
                jobId,
                userId: user.id,
                email,
                external,
                externalWebhook,
                generationStatus: 'pending',
                platform,
                voiceType,
                artStyle,
                aspectRatio,
                duration,
                uploadTime,
                postToYoutube: platform === 'youtube',
                prompt: promptToUse,
                enableCaptions,
                captionStyle: JSON.stringify(captionStyleDetails),
            });
            logger.info(`running refine prompt for job: ${jobId}`)
            const imagePrompt = await OpenAIService.refinePrompt(promptToUse, artStyle);

            logger.info(`running generating audio for job: ${jobId}`)
            const { audioURL, alignment } = await ElevenLabsService.generateAudio(promptToUse, voiceType);
            
            logger.info(`running generating imageID for job: ${jobId}`)
            const generationId = await LeonardoService.generateImages(imagePrompt, artStyle, aspectRatio, duration);
            const sentenceTimings = parseElevenLabsTimestamps(alignment);
            const currentStep = 'step1';

            await Video.update({ generationId, audioURL, sentenceTimings: JSON.stringify(sentenceTimings), currentStep }, { where: { jobId } });

            res.status(200).json({ message: "Initial generation complete", jobId, audioURL, generationId, currentStep });

        } catch (error) {
            logger.error(`Error in initial generation: ${error}`);
            res.status(500).json({
                error: 'Failed to generate initial components',
                details: error,
            });
        }
    }

    async waitTimePromise(req, res) {
        try {
            const { jobId } = req.body;
            if (!jobId) {
                return res.status(400).json({
                    error: `Job Id must be provided`,
                });
            }
            const video = await Video.findOne({ where: { jobId } });

            if (!video) return res.status(404).json({ error: 'Job not found' });
            if (video.currentStep !== 'step1') {
                return res.status(400).json({ error: 'Invalid step. The job must be in step 1 before proceeding.' });
            }
            logger.info(`running leonardo wait time for job: ${jobId}`)
            await new Promise((resolve) => setTimeout(resolve, 20000)); 
            const currentStep = 'step2'
            await Video.update({ currentStep }, { where: { jobId } });
            res.status(200).json({ message: "Wait Time completed", jobId, currentStep });
        } catch (error) {
            logger.error(`Error running wait time: ${error.message}`);
            res.status(500).json({ error: 'Failed to run wait time', details: error.message });
        }
    }

    async getGeneratedImages(req, res) {
        try {
            const { jobId } = req.body;
            if (!jobId) {
                return res.status(400).json({
                    error: `Job Id must be provided`,
                });
            }
            const video = await Video.findOne({ where: { jobId } });
    
            if (!video) return res.status(404).json({ error: 'Job not found' });
    
            // Ensure the job is at step2 before proceeding
            if (video.currentStep !== 'step2') {
                return res.status(400).json({ error: 'Invalid step. The job must be in step 2 before proceeding.' });
            }
    
            logger.info(`Running getGeneratedImages for job: ${jobId}`);
            const images = await LeonardoService.getGeneratedImages(video.generationId);
    
            // console.log(JSON.stringify(images))
            const currentStep = 'step3';
            await Video.update(
                { images: JSON.stringify(images), currentStep },
                { where: { jobId } }
            );
    
            res.status(200).json({ message: "Images generated", jobId, currentStep });
        } catch (error) {
            logger.error(`Error fetching generated images: ${error}`);
            res.status(500).json({ error: 'Failed to retrieve images', details: error });
        }
    }
    

    // async generateVideoInBackground(jobId) {
    //     try {
    //         const video = await Video.findOne({ where: { jobId } });
    //         if (!video) throw new Error('Video not found');
    
    //         const email = video.email;
    //         const platform = video.platform;
    //         const user = await User.findOne({ where: { email } });
    
    //         logger.info(`Running video creation for job: ${jobId}`);
    //         const videoPath = await ffmpegService.generateVideo({
    //             imagePaths: video.images,
    //             audioURL: video.audioURL,
    //             sentenceTimings: video.sentenceTimings,
    //         });
    //         logger.info(`FFmpeg completed for job: ${jobId}`);
    
    //         await Video.update({ generationStatus: 'completed', videoPath }, { where: { jobId } });
    
    //         let uploadError = false;
    
    //         if (platform === 'youtube') {
    //             if (user?.googleTokens) {
    //                 try {
    //                     const videoId = await youtubeService.uploadVideo(
    //                         user.googleTokens.access_token,
    //                         user.googleTokens.refresh_token,
    //                         videoPath,
    //                         videoMetadata
    //                     );
    //                     await sendVideoReadyEmail({
    //                         to: email,
    //                         jobId,
    //                         message: `Your video is available on YouTube: https://www.youtube.com/watch?v=${videoId}`,
    //                         videoPath,
    //                     });
    //                 } catch (uploadError) {
    //                     await sendVideoReadyEmail({
    //                         to: email,
    //                         jobId,
    //                         message: `Your video has been successfully generated, but there was an error posting to YouTube. Our team has been notified.`,
    //                         videoPath,
    //                     });
    //                     logger.error(`YouTube upload failed for job ${jobId}: ${uploadError.message}`);
    //                     uploadError = true;
    //                 }
    //             } else {
    //                 logger.error(`Google tokens missing for user ${email}. Cannot upload to YouTube.`);
    //                 uploadError = true;
    //             }
    //         }
    
    //         if (platform === 'email') {
    //             try {
    //                 await sendVideoReadyEmail({
    //                     to: email,
    //                     jobId,
    //                     message: `Your video has been generated. Please find it attached.`,
    //                     videoPath,
    //                 });
    //             } catch (emailError) {
    //                 logger.error(`Email sending failed for job ${jobId}: ${emailError.message}`);
    //                 uploadError = true;
    //             }
    //         }
    
    //         if (uploadError) {
    //             await Video.update({ isPostSuccessful: false }, { where: { jobId } });
    //         } else {
    //             await Video.update({ isPostSuccessful: true }, { where: { jobId } });
    //         }
    
    //         logger.info(`Video generation completed for job ${jobId}`);
    //     } catch (error) {
    //         logger.error(`Error generating video: ${error.message}`);
    //         await sendVideoGenerationFailedEmail({ to: email, jobId, errorMessage: 'RES2732' });
    //         await sendFailedVideotoAdmin({ user: email, jobId, errorMessage: error.message });
    //     }
    // }

    async generateVideo(req, res) {
        console.log("generateVideo called");
        const {
             jobId,
              email = null
             } = req.body;
        if (!jobId) {
            return res.status(400).json({ error: `Job Id must be provided` });
        }
    
        const video = await Video.findOne({ where: { jobId } });
        if (!video) return res.status(404).json({ error: 'Video not found' });

        if (video.currentStep !== 'step3') {
            return res.status(400).json({ error: 'Invalid step. The job must be in step 3 before proceeding.' });
        } 
        const currentStep = "step4";
        await Video.update({ currentStep }, { where: { jobId } });
        // Immediately respond to the frontend
        res.status(200).json({ message: "Video generation started. You will be notified via email when it's ready.", jobId, currentStep });
    
        // Start the video generation process in the background
        generateVideoInBackground(jobId, email).catch((error) => {
            logger.error(`Background video generation failed for job ${jobId}: ${error.message}`);
        });
        

    }
    
    async regenerateVideo(req, res) {
        const { jobId } = req.params;
        const userId = req.user.id;

        try {
            const user = await User.findByPk(userId);
            const email = user.email;

            const video = await Video.findOne({ where: { jobId } });

            if (!video) {
                return res.status(404).json({ error: 'Video not found' });
            }

            if (userId !== video.userId) {
                return res.status(403).json({ error: 'You are not allowed to regenerate this video' });
            }

            if (video.generationStatus !== 'failed') {
                return res.status(400).json({
                    error: 'Video cannot be regenerated unless its status is "failed".',
                });
            }

            // Update video status to pending
            await Video.update(
                {
                    generationStatus: 'pending',
                },
                { where: { jobId } }
            );

            // Prepare video details for regeneration
            const videoDetails = {
                voiceType: video.voiceType,
                artStyle: video.artStyle,
                aspectRatio: video.aspectRatio,
                duration: video.duration,
                platform: video.platform,
                email: video.email,
                prompt: video.prompt,
            };

            // Call the video generation logic synchronously
            await processVideoGenerationJob({ data: { jobId, videoDetails } });

            // Send response after video regeneration is complete
            const result = await processVideoGenerationJob({ data: { jobId, videoDetails } });

            if (result.success) {
                res.status(200).json({ message: result.message, jobId, videoPath: result.videoPath });
            } else {
                res.status(500).json({ error: result.message, details: result.error });
            }
        } catch (error) {
            logger.error(`Error regenerating video for jobId ${jobId}: ${error.message}`);
            res.status(500).json({ error: 'Failed to regenerate video', details: error.message });
        }
    }

    async generateCustomPrompt(req, res) {
        try {
            const customPrompt = await OpenAIService.generateCustomPrompt();

            res.json({
                message: 'Custom prompt generated successfully',
                prompt: customPrompt,
            });
        } catch (error) {
            logger.error(`Custom prompt generation failed: ${error.message}`);
            res.status(500).json({
                error: 'Custom prompt generation failed',
                details: error.message,
            });
        }
    }

    async generateCustomPromptBasedOnAnIdea(req, res) {
        try {
            const { prompt, duration, character } = req.body;
            const customPrompt = await OpenAIService.generateVoiceoverScript(prompt, duration, character);

            res.json({
                message: 'Custom prompt generated successfully',
                prompt: customPrompt,
            });
        } catch (error) {
            logger.error(`Custom prompt generation failed: ${error.message}`);
            res.status(500).json({
                error: 'Custom prompt generation failed',
                details: error.message,
            });
        }
    }

    async moderateText(req, res) {
        try {
            const { prompt } = req.body;
            const results = await OpenAIService.moderateText(prompt);

            res.json({
                message: 'Text checked successfully',
                results,
            });
        } catch (error) {
            logger.error(`Text check failed: ${error.message}`);
            res.status(500).json({
                error: 'Text check failed',
                details: error.message,
            });
        }
    }

    async getUserProgress(req, res) {
        try {
            const { id } = req.user;
            const latestJob = await Video.findOne({ 
                where: { userId: id, currentStep: { [Op.ne]: 'done' } }
            });
    
            if (!latestJob) {
                return res.status(200).json({ message: 'No ongoing process!.' });
            }
    
            res.status(200).json({ latestJob: latestJob.jobId, currentStep: latestJob.currentStep });
        } catch (error) {
            res.status(500).json({ error: 'Failed to fetch progress' });
        }
    }
    
    async markJobAsDone(req, res) {
        try {
            const { jobId } = req.body;
            if (!jobId) {
                return res.status(400).json({ error: 'Job Id must be provided' });
            }
            const video = await Video.findOne({ where: { jobId } });
            if (!video) {
                return res.status(404).json({ error: 'Job not found' });
            }
            await Video.update({ currentStep: 'done' }, { where: { jobId } });
            res.status(200).json({ message: 'Job marked as done', jobId, currentStep: 'done' });
        } catch (error) {
            logger.error(`Error marking job as done: ${error.message}`);
            res.status(500).json({ error: 'Failed to mark job as done', details: error.message });
        }
    }
}

module.exports = new VideoController();