const express = require('express');
const { check } = require('express-validator');
const authController = require('../controllers/authController');
const { authMiddleware } = require('../middlewares/authMiddleware');
const { handlePasswordResetRequest, resetUserPassword } = require('../controllers/authResetController');
const { verifyEmail, resendVerificationEmail } = require('../controllers/emailVerificationController');

const router = express.Router();

// Register Route
router.post(
  '/admin-register',
  [
    check('name', 'Full name is required').notEmpty(),
    check('email', 'Please include a valid email').isEmail(),
    check('password', 'Password must be at least 6 characters').isLength({ min: 6 }),
  ],
  authController.createAdmin
);

// Login Route
router.post('/login', authController.login);

// Reseller Registration Route
router.post(
  '/reseller-register',
  [
    check('name', 'Full name is required').notEmpty(),
    check('email', 'Please include a valid email').isEmail(),
    check('password', 'Password must be at least 6 characters').isLength({ min: 6 }),
  ],
  authController.createReseller
);

// Resend Verification Email
router.post('/resend-verification', resendVerificationEmail);

// Verify Email
router.get('/verify-email/:token', verifyEmail);

// Protected Routes
router.get('/profile', authMiddleware, authController.getProfile);

router.get('/generate-api-key', authMiddleware, authController.generateApiKey);

// Update User Route
router.put(
  '/profile/update',
  authMiddleware,
  [
    check('name', 'Full name is required').optional().notEmpty(),
    check('email', 'Please include a valid email').optional().isEmail(),
    check('password', 'Password must be at least 6 characters').optional().isLength({ min: 6 }),
  ],
  authController.updateUser
);

// Delete User Route
router.delete('/profile/delete', authMiddleware, authController.deleteUser);

// / Password Reset Request Route
router.post('/password-reset', async (req, res) => {
  const { email } = req.body;

  try {
    await handlePasswordResetRequest(email);
    return res.status(200).json({ message: 'Password reset link sent successfully' });
  } catch (err) {
    return res.status(500).json({ error: err.message });
  }
});

// Password Reset Confirmation Route
router.post('/password-reset/confirm', async (req, res) => {
  const { token, password, passwordConfirmation } = req.body;

  try {
    await resetUserPassword(token, password, passwordConfirmation);
    return res.status(200).json({ message: 'Password has been reset successfully' });
  } catch (err) {
    return res.status(500).json({ error: err.message });
  }
});


module.exports = router;
